C      *****************************************************************
C      * Subroutine SoilChem                                           *
C      * This is the top level controlling routine for the soil        *
C      * chemistry model.  It handles initialization, including input  *
C      * of model parameters and calculation of soil chemical          *
C      * concentrations.                                               *
C      *                                                               *
C      * Variables:                                                    *
C      * DcmpDetail-Input - Indicates whether detailed output for      *
C      *                    the decomposition routine should be written*
C      *                    (TRUE) or not (FALSE).  Unitless.          *
C      * DecompFile-Local - The name of the file for output of decom-  *
C      *                    position details.  Unitless.               *
C      * FName     -Input - The name of the file to be read in by      *
C      *                    the routine InDat. Input from TREGRO, or   *
C      *                    read into the model by the routine InParam.*
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * FrsTim    -Input - A flag that is a 1 the first time the soil *
C      *                    model is called and is a 0 otherwise.      *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VModel, file VModel.f)       *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * N         -Local - Layer number index.  Unitless.             *
C      * NComp     -Global- The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * N_Index   -Output- The index number for the component whose   *
C      *                    known total is to be updated by the        *
C      *                    mineralized nitrogen.  Unitless.           *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * NitInor   -Output- A vector of size NLayers, of the amount of *
C      *                    inorganic soil nitrogen  per layer.  Units *
C      *                    are g N.                                   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * OutInfo   -Output- Indicates whether or not detailed          *
C      *                    information is output.  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *              
C      * StndAlone -Input - Indicates whether soil model is running as *
C      *                    a stand alone model (TRUE) or not (FALSE). * 
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * Tab       -Local - The tab character.  Unitless.              *
C      * UseDecomp -In/Out- Indicates whether decomposition is to be   *
C      *                    run (TRUE) or not (FALSE).  Unitless.      *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE SOILCHEM(GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							INTEGER GOAHEAD, N
       CHARACTER*1 TAB
       CHARACTER*255 DECOMPFILE

C      *------------------------*
C      * Set the tab character. *
C      *------------------------*
       TAB = CHAR(9)

C      *----------------------------------------------------------------*
C      * Initialize variable to indicate that no errors have been found *
C      *----------------------------------------------------------------*
       GOAHEAD = 1
	   
C      *-------------------------------------------------------------*
C      * If this is the first time that this routine has been called *
C      * set up startup parameters and check some model input values.*
C      *-------------------------------------------------------------*
       IF (FRSTIM.EQ.1) THEN 
          CALL STARTUP(GOAHEAD)
       ENDIF
		 
C      *-----------------------------------------------------------------*
C      * Continue with the soil chemistry model as long as all interface *
C      * species are present or the user has chosen to continue          *
C      *-----------------------------------------------------------------*
       IF (GOAHEAD.NE.1) GOTO 9999

C      *--------------------------------------------------------*
C      * Set up the multiplication factors used to convert from *
C      * concentration to mass.                                 *
C      *--------------------------------------------------------*
       CALL INITCTOM

C      *---------------------------------------------------------------*
C      * Overwrite the current throughfall concentrations for species  *
C      * of interest with the input values from the tree model.        *
C      *---------------------------------------------------------------*
       IF (.NOT.STNDALONE) THEN
          CALL TFSETUP
       ENDIF

C      *---------------------------------------------------------------*
C      * If this is the first time that YASE has been called, check to *
C      * see if the decomposition routine is being run and whether a   *
C      * detailed output file is to be made.  If so, open the file and *
C      * write the header.                                             *
C      *---------------------------------------------------------------*
       IF (FRSTIM.EQ.1) THEN

C         *-------------------------------------------------------*
C         * If detailed output for decomposition is to be output, *
C         * open the decomposition output data file. Substitute   *
C         * ".Dec". for ".Out".                                   *
C         *-------------------------------------------------------*
          IF (USEDECOMP.AND.DCMPDETAIL) THEN
	            DCMPOUT = 11
             N = INDEX(OUTFILE, '    ')
             DECOMPFILE = OUTFILE(:N-5) // '.DEC'
             WRITE(6,*) 'Creating decomp. output file: ', DECOMPFILE
	            OPEN(UNIT = DCMPOUT, STATUS='NEW', FILE = DECOMPFILE)
             WRITE(DCMPOUT, 101) 'LYR',TAB, 'TOTRESP',TAB, 'NETMIN',TAB,
     >       'EC',TAB, 'CC',TAB, 'LC',TAB, 'HC',TAB, 'EN',TAB, 'N',TAB 
             WRITE(DCMPOUT, 101) '#', TAB, '(gC/timestep)', TAB,  
     >       '(gN/timestep)', TAB, '(gC)', TAB, '(gC)', TAB, '(gC)', 
     >        TAB, '(gC)', TAB, '(gN)', TAB, '(gN)', TAB
  101        FORMAT(' ', 25(A20,A1))
	         ENDIF 

C      *-----------------------------------------------------------------*
C      * If this is not the first time, calculate new known totals, then *
C      * equilibrate the system.  If it is the first time, just          *
C      * equilibrate the system.                                         *
C      *-----------------------------------------------------------------*
       ELSE
	     
C         *-------------------------------------------------------------*
C         * Update the known concentrations of the components based on  *
C         * nutrient uptake by the tree, weathering, water movement and *
C         * decomposition.                                              *
C         *-------------------------------------------------------------*
          IF (.NOT.STNDALONE) THEN
             CALL NUTREE(GOAHEAD)
		           IF (GOAHEAD.NE.1) GOTO 9999
          ENDIF
          IF (USEDECOMP) CALL DECNTRL
          CALL WTHRKT(GOAHEAD)
		        IF (GOAHEAD.NE.1) GOTO 9999
	         CALL FLUXKT(GOAHEAD)
		        IF (GOAHEAD.NE.1) GOTO 9999
	      ENDIF
	   
C      *--------------------------------------------*
C      * Equilibrate the soil system layer by layer *
C      *--------------------------------------------*
       DO 100 N=1, NLAYERS
		  
C         *-----------------------------------------------------------*
C         * If this is the first time through, determine a first      *
C         * guess for the component concentrations for this layer,    *
C         * otherwise use the concentrations from the last time step. *
C         *-----------------------------------------------------------*
	         IF (FRSTIM.EQ.1) CALL FGUESS(N)
	   
C         *---------------------------------------------*
C         * Solve for the concentrations of all species *
C         *---------------------------------------------*
          CALL SOLVE(N, YASOUT, OUTINFO, GOAHEAD, NCOMP)
          IF (GOAHEAD.NE.1) GOTO 9999
  100  CONTINUE

C      *--------------------*
C      * Error escape hatch *
C      *--------------------*
 9999  CONTINUE	 
  
       RETURN
	      END
C      *****************************************************************
C      *                   END SUBROUTINE                              *
C      *****************************************************************
